<?php
declare(strict_types=1);

function handleInlineButtonTap(string $endpoint, array $update, int $chatId, string $botToken): bool
{
    $callbackQuery = $update['callback_query'] ?? null;
    if (!$callbackQuery) {
        return false;
    }

    $data = $callbackQuery['data'] ?? '';
    if ($data === '') {
        return false;
    }

    $senderId = $callbackQuery['from']['id'] ?? null;
    $originMessageId = $callbackQuery['message']['message_id'] ?? null;

    // معالجة أزرار الإدارة
    if (strpos($data, 'admin_') === 0 && $senderId !== null) {
        if (function_exists('isAdmin') && function_exists('handleAdminInlineButtons')) {
            if (isAdmin($senderId)) {
                return handleAdminInlineButtons($endpoint, $chatId, $data, $senderId, $originMessageId, $botToken, $update);
            }
        }
    }

    $responses = [
        'action_search_help' => <<<TXT
🔍 للبحث عن هدية:
- انسخ رابط الهدية من تيليجرام (مثال: https://t.me/nft/IceCream-103931).
- أرسله هنا وسيعرض لك البوت تفاصيلها وأسعارها المختلفة.
TXT,
        'action_convert_help' => <<<TXT
💱 تصريف العملات:
- اكتب الكمية ونوع العملة مثل 1 TON، أو 50 نجمة، أو 2 آسيا.
- البوت سيبين لك السعر بالدولار، وما يعادله بـ TON، وآسيا، وماستر، وعدد النجوم.
TXT,
        'action_commands' => <<<TXT
📋 <b>جميع الأوامر المتاحة:</b>

🔍 <b>البحث عن هدايا:</b>
- أرسل رابط الهدية من تيليجرام
- مثال: https://t.me/nft/IceCream-103931

💱 <b>تصريف العملات:</b>
- اكتب الكمية ونوع العملة
- أمثلة: 1 TON، 50 نجمة، 2 آسيا، 100 دولار، 5 ماستر

💸 <b>التحويل:</b>
- اكتب كلمة "تحويل" لإنشاء كود تحويل
- سيطلب منك البوت المبلغ ثم الرقم

📋 <b>الأوامر:</b>
- اضغط على زر "الأوامر" لعرض جميع الأوامر المتاحة
TXT,
    ];

    if (isset($responses[$data])) {
        answerCallbackQuery($endpoint, $callbackQuery['id'] ?? '');

        if ($originMessageId !== null) {
            sendMessage(
                $endpoint,
                $chatId,
                $responses[$data],
                ['reply_markup' => json_encode(buildMainInlineKeyboard(), JSON_UNESCAPED_UNICODE)]
            );
            return true;
        }
    }

    return false;
}

function buildMainInlineKeyboard(): array
{
    return [
        'inline_keyboard' => [
            [
                ['text' => '🔍 البحث عن هدايا', 'callback_data' => 'action_search_help'],
                ['text' => '💱 تصريف العملات', 'callback_data' => 'action_convert_help'],
            ],
            [
                ['text' => '📋 الأوامر', 'callback_data' => 'action_commands'],
            ],
        ],
    ];
}

const STAR_USD_VALUE = 0.015;
const ASIA_MULTIPLIER = 1.6;
const MASTER_MULTIPLIER = 1.45;
const CONVERSION_STATE_DIR = __DIR__ . DIRECTORY_SEPARATOR . 'conversion_state';
const ADMIN_DATA_DIR = __DIR__ . DIRECTORY_SEPARATOR . 'admin_data';
const ADMIN_DATA_FILE = ADMIN_DATA_DIR . DIRECTORY_SEPARATOR . 'admin.json';
const USERS_DATA_FILE = ADMIN_DATA_DIR . DIRECTORY_SEPARATOR . 'users.json';
const GROUPS_DATA_FILE = ADMIN_DATA_DIR . DIRECTORY_SEPARATOR . 'groups.json';
const CHANNELS_DATA_FILE = ADMIN_DATA_DIR . DIRECTORY_SEPARATOR . 'channels.json';
const GROUP_CHANNELS_FILE = ADMIN_DATA_DIR . DIRECTORY_SEPARATOR . 'group_channels.json';

$botToken = '1873000913:AAHLp24J9XSmnJHNI27EHjV6uCAWXrupVdM';

if (!$botToken) {
    error_log('TELEGRAM_BOT_TOKEN is not set. Please configure the environment variable before using the bot.');
    http_response_code(500);
    exit;
}

$apiEndpoint = "https://api.telegram.org/bot{$botToken}/sendMessage";
$rawInput = file_get_contents('php://input') ?: '';

if ($rawInput === '') {
    http_response_code(204);
    exit;
}

$update = json_decode($rawInput, true);

if (!$update || json_last_error() !== JSON_ERROR_NONE) {
    error_log('Telegram Bot Error: Invalid JSON update - ' . json_last_error_msg() . ' - Input: ' . substr($rawInput, 0, 200));
    http_response_code(400);
    exit;
}

$callbackQuery = $update['callback_query'] ?? null;
$message = $callbackQuery['message'] ?? $update['message'] ?? $update['edited_message'] ?? null;

if (!$message) {
    http_response_code(204);
    exit;
}

$chatId = $message['chat']['id'] ?? null;
$chatType = $message['chat']['type'] ?? 'private';
$senderId = $callbackQuery['from']['id'] ?? $message['from']['id'] ?? null;
$text = trim($callbackQuery['data'] ?? $message['text'] ?? '');
$messageId = $message['message_id'] ?? null;
$from = $callbackQuery['from'] ?? $message['from'] ?? null;

if (!$chatId) {
    http_response_code(400);
    exit;
}

// تهيئة بيانات الإدارة
initAdminData();

// تسجيل المستخدمين والكروبات
if ($senderId !== null && $from !== null) {
    try {
        $userInfo = [
            'username' => $from['username'] ?? null,
            'first_name' => $from['first_name'] ?? null,
            'last_name' => $from['last_name'] ?? null,
        ];
        $oldUsers = loadUsersData();
        $isNewUser = !isset($oldUsers[$senderId]);
        addUser($senderId, $userInfo);
        
        // إرسال إشعار للمالك عند دخول مستخدم جديد
        if ($isNewUser && $chatType === 'private') {
            try {
                $adminData = loadAdminData();
                $users = loadUsersData();
                $userCount = count($users);
                
                // إرسال الإشعار دائماً للمالك حتى لو كانت الإشعارات معطلة
                // الإشعارات العامة تُستخدم لأشياء أخرى
                if (isset($adminData['owner']) && $adminData['owner'] > 0) {
                    $notification = "🔔 <b>مستخدم جديد</b>\n\n";
                    $notification .= "👤 <b>الاسم:</b> " . ($userInfo['first_name'] ?? 'غير معروف');
                    if (isset($userInfo['last_name'])) {
                        $notification .= " " . $userInfo['last_name'];
                    }
                    $notification .= "\n";
                    if (isset($userInfo['username'])) {
                        $notification .= "📱 <b>المعرف:</b> @" . $userInfo['username'] . "\n";
                    }
                    $notification .= "🆔 <b>الرقم:</b> " . $senderId . "\n";
                    $notification .= "👥 <b>إجمالي المستخدمين:</b> " . $userCount;
                    
                    $endpoint = "https://api.telegram.org/bot{$botToken}/sendMessage";
                    sendMessage($endpoint, $adminData['owner'], $notification);
                }
            } catch (Exception $e) {
                error_log("Telegram Bot Error: Failed to send new user notification - " . $e->getMessage());
            }
        }
    } catch (Exception $e) {
        // تجاهل أخطاء تسجيل المستخدمين
    }
}

// تسجيل الكروبات
if ($chatType === 'group' || $chatType === 'supergroup') {
    $groupInfo = [
        'title' => $message['chat']['title'] ?? null,
        'type' => $chatType,
    ];
    addGroup($chatId, $groupInfo);
}

if ($text === '' || $text === '/start') {
    if ($chatType === 'private') {
        // التحقق من الاشتراك في القنوات العامة (للخاص فقط)
        try {
            $adminData = loadAdminData();
            $privateChannels = $adminData['private_channels'] ?? [];
            if (!empty($privateChannels) && $senderId !== null) {
                $notSubscribed = checkChannelSubscription($botToken, $senderId, $privateChannels);
                if (!empty($notSubscribed)) {
                    $channelsList = implode(', ', array_map(fn($id) => '@' . ltrim($id, '@'), $notSubscribed));
                    sendMessage(
                        $apiEndpoint,
                        $chatId,
                        "⚠️ يجب الاشتراك في القنوات التالية أولاً:\n" . $channelsList,
                        $messageId !== null ? ['reply_to_message_id' => $messageId] : []
                    );
                    exit;
                }
            }
        } catch (Exception $e) {
            // تجاهل الأخطاء في التحقق من الاشتراك والمتابعة
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔍 البحث عن هدايا', 'callback_data' => 'action_search_help'],
                    ['text' => '💱 تصريف العملات', 'callback_data' => 'action_convert_help'],
                ],
                [
                    ['text' => '📋 الأوامر', 'callback_data' => 'action_commands'],
                ],
            ],
        ];

        $welcome = <<<TXT
👋 أهلاً بك في بوت هدايا!

اختر أحد الأزرار أو اتبع التعليمات:

🔍 البحث عن هدايا:
- أرسل رابط الهدية من تيليجرام مثل https://t.me/nft/اسم-الهدية-1234
- سيعرض لك البوت تفاصيل الهدية وأسعارها.

💱 تصريف العملات:
- اكتب الكمية مع نوع العملة، مثال: 10 TON أو 50 نجوم أو 2 آسيا.
- سيعرض البوت السعر بالدولار، وTON، وآسيا، وماستر، وعدد النجوم.
TXT;

        sendMessage(
            $apiEndpoint,
            $chatId,
            $welcome,
            [
                'reply_markup' => json_encode(buildMainInlineKeyboard(), JSON_UNESCAPED_UNICODE),
            ]
        );
    }
    exit;
}

// معالجة أوامر الإدارة
if (handleAdminCommands($apiEndpoint, $chatId, $text, $senderId, $messageId, $botToken, $update)) {
    exit;
}

// التحقق من الاشتراك في قنوات الكروبات (للأوامر في الكروبات)
if (($chatType === 'group' || $chatType === 'supergroup') && $senderId !== null) {
    $groupChannels = loadGroupChannelsData();
    if (!empty($groupChannels)) {
        $notSubscribed = checkChannelSubscription($botToken, $senderId, $groupChannels);
        if (!empty($notSubscribed)) {
            $channelsList = implode(', ', array_map(fn($id) => '@' . ltrim($id, '@'), $notSubscribed));
            sendMessage(
                $apiEndpoint,
                $chatId,
                "⚠️ يجب الاشتراك في القنوات التالية لاستخدام البوت:\n" . $channelsList,
                $messageId !== null ? ['reply_to_message_id' => $messageId] : []
            );
            exit;
        }
    }
}

// التحقق من الاشتراك في القنوات العامة (للخاص فقط)
if ($chatType === 'private' && $senderId !== null) {
    $adminData = loadAdminData();
    $privateChannels = $adminData['private_channels'] ?? [];
    if (!empty($privateChannels)) {
        $notSubscribed = checkChannelSubscription($botToken, $senderId, $privateChannels);
        if (!empty($notSubscribed)) {
            $channelsList = implode(', ', array_map(fn($id) => '@' . ltrim($id, '@'), $notSubscribed));
            sendMessage(
                $apiEndpoint,
                $chatId,
                "⚠️ يجب الاشتراك في القنوات التالية أولاً:\n" . $channelsList,
                $messageId !== null ? ['reply_to_message_id' => $messageId] : []
            );
            exit;
        }
    }
}

if (handleTransferWorkflow($apiEndpoint, $chatId, $text, $senderId, $messageId)) {
    exit;
}

if ($chatType === 'private' && handleInlineButtonTap($apiEndpoint, $update, $chatId, $botToken)) {
    exit;
}

if (tryHandleConversionMessage($apiEndpoint, $chatId, $text, $messageId)) {
    exit;
}

$giftQuery = extractGiftQuery($text);
if ($giftQuery === null) {
    exit;
}

try {
    $gifts = fetchGifts($giftQuery);
} catch (RuntimeException $e) {
    error_log("Telegram Bot Error: Failed to fetch gifts - " . $e->getMessage());
    sendMessage($apiEndpoint, $chatId, "❗ حدث خطأ أثناء جلب البيانات. حاول مجدداً لاحقاً.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
    exit;
}

if (count($gifts) === 0) {
    sendMessage($apiEndpoint, $chatId, "🔍 لم يتم العثور على هدايا تطابق الرابط المرسل.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
    exit;
}

$topGifts = array_slice($gifts, 0, 5);
$messages = array_map(
    fn(array $gift, int $index) => formatGiftMessage($gift, $index + 1),
    $topGifts,
    array_keys($topGifts)
);

$responseText = implode("\n\n", $messages);

sendMessage($apiEndpoint, $chatId, $responseText, $messageId !== null ? ['reply_to_message_id' => $messageId] : []);

exit;

function fetchGifts(string $query): array
{
    $url = 'https://giftcharts-api.onrender.com/gifts';
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($response === false || $error || ($httpCode !== 200 && $httpCode !== 0)) {
        error_log("Telegram Bot Error: Gift API request failed - HTTP: {$httpCode}, Error: {$error}");
        throw new RuntimeException('Gift API request failed.');
    }

    $data = json_decode($response, true);
    if (!is_array($data)) {
        error_log("Telegram Bot Error: Invalid response from Gift API - " . substr($response ?: '', 0, 200));
        $data = [];
    }

    $filtered = array_filter(
        $data,
        fn(array $gift) => stripos($gift['name'] ?? '', $query) !== false
    );

    return array_values($filtered);
}

function extractGiftQuery(string $text): ?string
{
    $text = trim($text);

    if ($text === '') {
        return null;
    }

    $pattern = '~(?:https?://)?t\.me/nft/([^?\s#]+)~i';
    if (preg_match($pattern, $text, $matches)) {
        $slug = $matches[1];
        $slug = strtok($slug, '?');
        $parts = explode('-', $slug);

        if (count($parts) > 1 && preg_match('/^\d+$/', end($parts))) {
            array_pop($parts);
        }

        $core = implode('-', $parts);
        $core = str_replace('-', ' ', $core);
        $core = preg_replace('/(?<!^)([A-Z])/u', ' $1', $core);
        $core = preg_replace('/\s+/', ' ', $core);

        return trim($core);
    }

    return null;
}

function formatGiftMessage(array $gift, int $position): string
{
    $name = $gift['name'] ?? 'غير معروف';
    $supply = $gift['supply'] ?? 'غير متوفر';
    $priceUsd = $gift['priceUsd'] ?? 'غير متوفر';
    $priceTon = $gift['priceTon'] ?? 'غير متوفر';
    $releaseDate = $gift['releaseDate'] ?? 'غير متوفر';
    $starsPrice = $gift['starsPrice'] ?? null;
    $asiaPrice = null;
    $masterPrice = null;
    $starsCount = null;
    if (is_numeric($priceUsd) && STAR_USD_VALUE > 0) {
        $starsCount = (int)round((float)$priceUsd / STAR_USD_VALUE);
        $asiaPrice = round((float)$priceUsd * ASIA_MULTIPLIER, 6);
        $masterPrice = round((float)$priceUsd * MASTER_MULTIPLIER, 6);
    } elseif (is_numeric($starsPrice)) {
        $starsCount = (int)round((float)$starsPrice);
    }

    $lines = [
        "{$position}. 🎁 {$name}",
        "📦 المخزون الحالي: {$supply}",
        "💵 السعر بالدولار: {$priceUsd}",
        "🪙 السعر بـ TON: {$priceTon}",
        "📅 تاريخ الإصدار: {$releaseDate}",
    ];

    if ($starsCount !== null) {
        $lines[] = "⭐️ السعر بالنجوم: " . formatStars($starsCount) . " نجمة";
    }

    if ($asiaPrice !== null) {
        $lines[] = "🌏 السعر في آسيا: {$asiaPrice} دولار";
    }

    if ($masterPrice !== null) {
        $lines[] = "👑 السعر في ماستر: {$masterPrice} دولار";
    }

    return implode("\n", $lines);
}

function tryHandleConversionMessage(string $endpoint, int $chatId, string $text, ?int $replyToMessageId = null): bool
{
    $pattern = '~^\s*([\d.,]+)\s*(ton|تون|t|usd|us|usdt|يو اس دي تي|دولار|دولار امريكي|asia|آسيا|اسيا|as|ماستر|master|ms|نجمة|نجمه|نجوم|star|stars)\s*$~iu';
    if (!preg_match($pattern, $text, $matches)) {
        return false;
    }

    $amountString = str_replace(',', '.', $matches[1]);
    if (!is_numeric($amountString)) {
        sendMessage($endpoint, $chatId, "❗ القيمة المدخلة غير صالحة للتحويل.", $replyToMessageId !== null ? ['reply_to_message_id' => $replyToMessageId] : []);
        return true;
    }

    $amount = (float)$amountString;
    $rawCurrencyLabel = $matches[2];
    $currency = normalizeCurrencyLabel($rawCurrencyLabel);

    try {
        $conversion = convertCurrencyAmount($amount, $currency, $rawCurrencyLabel);
        $usdValue = $conversion['usd'];
        $amountLabel = $conversion['label'];
    } catch (RuntimeException $e) {
        sendMessage($endpoint, $chatId, "❗ تعذر إكمال التحويل حالياً. حاول لاحقاً.", $replyToMessageId !== null ? ['reply_to_message_id' => $replyToMessageId] : []);
        return true;
    }

    $messageLines = [
        "<b>🔄 تفاصيل التحويل</b>",
        "• <b>القيمة الأصلية:</b> " . formatDecimal($amount) . " {$amountLabel}",
        "• <b>السعر بالدولار:</b> " . formatDecimal($usdValue) . " $",
        "• <b>ما يعادل بـ TON:</b> " . formatDecimal($conversion['ton']) . " TON",
        "• <b>السعر في آسيا:</b> " . formatDecimal($conversion['asia']) . " $",
        "• <b>السعر في ماستر:</b> " . formatDecimal($conversion['master']) . " $",
        "• <b>ما يعادل بالنجوم:</b> " . formatStars($conversion['stars']) . " نجمة",
    ];

    $message = implode("\n", $messageLines);

    $options = [];
    if ($replyToMessageId !== null) {
        $options['reply_to_message_id'] = $replyToMessageId;
    }
    sendMessage($endpoint, $chatId, $message, $options);
    return true;
}

function handleTransferWorkflow(string $endpoint, int $chatId, string $text, ?int $senderId, ?int $messageId = null): bool
{
    $trimmed = trim($text);
    if ($trimmed === '') {
        return false;
    }

    $lower = mb_strtolower($trimmed);

    if ($lower === 'تحويل') {
        $statePayload = ['step' => 'await_amount'];
        if ($senderId !== null) {
            $statePayload['user_id'] = $senderId;
        }
        saveTransferState($chatId, $statePayload);

        $options = [];
        if ($messageId !== null) {
            $options['reply_to_message_id'] = $messageId;
        }
        sendMessage(
            $endpoint,
            $chatId,
            <<<TXT
✅ حسناً عزيزي، قم بإرسال المبلغ الآن.
مثال: 25000
TXT
            ,
            $options
        );

        return true;
    }

    $state = loadTransferState($chatId);
    if ($state !== null && !isset($state['user_id']) && $senderId !== null) {
        $state['user_id'] = $senderId;
        saveTransferState($chatId, $state);
    }
    $stateStep = $state['step'] ?? null;
    $stateUserId = $state['user_id'] ?? null;

    $isDifferentUser = $state !== null
        && $stateUserId !== null
        && $senderId !== null
        && $senderId !== $stateUserId
        && in_array($stateStep, ['await_amount', 'await_phone'], true);

    if ($isDifferentUser) {
        return true;
    }

    if (preg_match('/^\s*(\d{6,})\s+(\d+)\s*$/u', $trimmed, $matches)) {
        $phone = $matches[1];
        $amount = $matches[2];
        if ($state !== null && $stateUserId !== null && $senderId !== null && $senderId === $stateUserId) {
            clearTransferState($chatId);
        }

        sendTransferCode($endpoint, $chatId, $amount, $phone, $messageId);
        return true;
    }

    if ($state === null) {
        return false;
    }

    $step = $state['step'] ?? '';

    if ($step === 'await_amount') {
        if (!preg_match('/^\d+$/', $trimmed)) {
            $options = [];
            if ($messageId !== null) {
                $options['reply_to_message_id'] = $messageId;
            }
            sendMessage(
                $endpoint,
                $chatId,
                "⚠️ الرجاء إرسال المبلغ كأرقام فقط. مثال: 25000",
                $options
            );
            return true;
        }

        $nextState = [
            'step' => 'await_phone',
            'amount' => $trimmed,
        ];
        if ($stateUserId !== null) {
            $nextState['user_id'] = $stateUserId;
        } elseif ($senderId !== null) {
            $nextState['user_id'] = $senderId;
        }

        saveTransferState($chatId, $nextState);

        $options = [];
        if ($messageId !== null) {
            $options['reply_to_message_id'] = $messageId;
        }
        sendMessage(
            $endpoint,
            $chatId,
            <<<TXT
✅ استلمت المبلغ. الآن أرسل الرقم (مثال: 07710036066).
TXT
            ,
            $options
        );
        return true;
    }

    if ($step === 'await_phone') {
        if (!preg_match('/^\d{6,}$/', $trimmed)) {
            $options = [];
            if ($messageId !== null) {
                $options['reply_to_message_id'] = $messageId;
            }
            sendMessage(
                $endpoint,
                $chatId,
                "⚠️ الرجاء إرسال الرقم بصيغة صحيحة. مثال: 07710036066",
                $options
            );
            return true;
        }

        $amount = $state['amount'] ?? '';
        if ($amount === '') {
            clearTransferState($chatId);
            $options = [];
            if ($messageId !== null) {
                $options['reply_to_message_id'] = $messageId;
            }
            sendMessage($endpoint, $chatId, "⚠️ حدث خطأ في العملية. أرسل كلمة \"تحويل\" للبدء من جديد.", $options);
            return true;
        }

        if ($stateUserId === null || $senderId === null || $senderId === $stateUserId) {
            clearTransferState($chatId);
        }
        sendTransferCode($endpoint, $chatId, $amount, $trimmed, $messageId);
        return true;
    }

    clearTransferState($chatId);
    return false;
}

function sendTransferCode(string $endpoint, int $chatId, string $amount, string $phone, ?int $replyToMessageId = null): void
{
    $amountClean = preg_replace('/\D+/', '', $amount) ?: $amount;
    $phoneClean = preg_replace('/\D+/', '', $phone) ?: $phone;
    $code = "*123*{$amountClean}*{$phoneClean}*1#";

    $options = [];
    if ($replyToMessageId !== null) {
        $options['reply_to_message_id'] = $replyToMessageId;
    }

    sendMessage(
        $endpoint,
        $chatId,
        <<<TXT
✅ تم إنشاء الكود المطلوب:
<code>{$code}</code>

انسخه واستخدمه مباشرة.
TXT
        ,
        $options
    );
}

function saveTransferState(int $chatId, array $state): void
{
    if (!is_dir(CONVERSION_STATE_DIR)) {
        @mkdir(CONVERSION_STATE_DIR, 0777, true);
    }

    $path = getTransferStatePath($chatId);

    try {
        $payload = json_encode($state, JSON_THROW_ON_ERROR | JSON_UNESCAPED_UNICODE);
        @file_put_contents($path, $payload, LOCK_EX);
    } catch (\JsonException $e) {
        // تجاهل أخطاء التخزين
    }
}

function loadTransferState(int $chatId): ?array
{
    $path = getTransferStatePath($chatId);
    if (!is_readable($path)) {
        return null;
    }

    $content = @file_get_contents($path);
    if ($content === false) {
        return null;
    }

    $data = json_decode($content, true);
    return is_array($data) ? $data : null;
}

function clearTransferState(int $chatId): void
{
    $path = getTransferStatePath($chatId);
    if (is_file($path)) {
        @unlink($path);
    }
}

function getTransferStatePath(int $chatId): string
{
    return CONVERSION_STATE_DIR . DIRECTORY_SEPARATOR . "{$chatId}.json";
}

function normalizeCurrencyLabel(string $label): string
{
    $label = trim(mb_strtolower($label));

    if (in_array($label, ['ton', 'تون', 't'], true)) {
        return 'ton';
    }
    $usdLabels = ['usd', 'us', 'usdt', 'يو اس دي تي', 'دولار', 'دولار امريكي'];
    if (in_array($label, $usdLabels, true)) {
        return 'usd';
    }
    if (in_array($label, ['asia', 'آسيا', 'اسيا', 'as'], true)) {
        return 'asia';
    }
    if (in_array($label, ['master', 'ماستر', 'ms'], true)) {
        return 'master';
    }
    if (in_array($label, ['نجمة', 'نجمه', 'نجوم', 'star', 'stars'], true)) {
        return 'stars';
    }
    return $label;
}

function convertCurrencyAmount(float $amount, string $currency, string $sourceLabel): array
{
    static $cachedTonRate = null;

    $ensureTonRate = static function () use (&$cachedTonRate) {
        if ($cachedTonRate === null) {
            $cachedTonRate = fetchTonUsdRate();
        }
        return $cachedTonRate;
    };

    $displayLabel = resolveDisplayLabel($currency, $sourceLabel);

    switch ($currency) {
        case 'ton':
            return convertFromUsd($amount * $ensureTonRate(), $displayLabel, $ensureTonRate(), $amount);
        case 'usd':
            return convertFromUsd($amount, $displayLabel, $ensureTonRate());
        case 'asia':
            return convertFromUsd($amount / ASIA_MULTIPLIER, $displayLabel, $ensureTonRate());
        case 'master':
            return convertFromUsd($amount / MASTER_MULTIPLIER, $displayLabel, $ensureTonRate());
        case 'stars':
            return convertFromUsd($amount * STAR_USD_VALUE, $displayLabel, $ensureTonRate());
        default:
            throw new RuntimeException('عملة غير مدعومة.');
    }
}

function convertFromUsd(float $usdValue, string $label, float $tonRate, ?float $explicitTon = null): array
{
    $asia = $usdValue * ASIA_MULTIPLIER;
    $master = $usdValue * MASTER_MULTIPLIER;
    $stars = STAR_USD_VALUE > 0 ? $usdValue / STAR_USD_VALUE : 0;
    $ton = $explicitTon ?? ($tonRate > 0 ? $usdValue / $tonRate : 0);

    return [
        'usd' => $usdValue,
        'asia' => $asia,
        'master' => $master,
        'stars' => $stars,
        'ton' => $ton,
        'label' => $label,
    ];
}

function resolveDisplayLabel(string $currency, string $sourceLabel): string
{
    $sourceLabel = trim($sourceLabel);
    $hasLatin = preg_match('/[a-z]/i', $sourceLabel) === 1;

    switch ($currency) {
        case 'ton':
            return $hasLatin ? 'TON' : ($sourceLabel !== '' ? $sourceLabel : 'تون');
        
        case 'usd':
            if (preg_match('/usdt/i', $sourceLabel) === 1) {
                return 'USDT';
            }
            if (preg_match('/^us$/i', $sourceLabel) === 1) {
                return 'USD';
            }
            if ($hasLatin) {
                return $sourceLabel !== '' ? strtoupper($sourceLabel) : 'USD';
            }
            return $sourceLabel !== '' ? $sourceLabel : 'دولار';
        
        case 'asia':
            if (preg_match('/^as$/i', $sourceLabel) === 1) {
                return $hasLatin ? 'Asia' : 'آسيا';
            }
            if ($hasLatin) {
                return $sourceLabel !== '' ? ucfirst(strtolower($sourceLabel)) : 'Asia';
            }
            return $sourceLabel !== '' ? $sourceLabel : 'آسيا';
        
        case 'master':
            if (preg_match('/^ms$/i', $sourceLabel) === 1) {
                return $hasLatin ? 'Master' : 'ماستر';
            }
            if ($hasLatin) {
                return $sourceLabel !== '' ? ucfirst(strtolower($sourceLabel)) : 'Master';
            }
            return $sourceLabel !== '' ? $sourceLabel : 'ماستر';
        
        case 'stars':
            return $hasLatin ? 'Stars' : ($sourceLabel !== '' ? $sourceLabel : 'نجوم');
        
        default:
            return $sourceLabel !== '' ? $sourceLabel : strtoupper($currency);
    }
}
function fetchTonUsdRate(): float
{
    $cacheFile = __DIR__ . DIRECTORY_SEPARATOR . 'ton_rate_cache.json';
    $cacheTtlSeconds = 60;
    $cachedRate = null;

    if (is_readable($cacheFile)) {
        $cachedContent = @file_get_contents($cacheFile);
        if ($cachedContent !== false) {
            $cachedData = json_decode($cachedContent, true);
            if (
                is_array($cachedData) &&
                isset($cachedData['rate'], $cachedData['timestamp']) &&
                is_numeric($cachedData['rate']) &&
                is_numeric($cachedData['timestamp']) &&
                (time() - (int)$cachedData['timestamp']) <= $cacheTtlSeconds
            ) {
                return (float)$cachedData['rate'];
            }
            if (isset($cachedData['rate']) && is_numeric($cachedData['rate'])) {
                $cachedRate = (float)$cachedData['rate'];
            }
        }
    }

    $url = 'https://api.coingecko.com/api/v3/simple/price?ids=the-open-network&vs_currencies=usd';
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if ($response === false || $error) {
        error_log("Telegram Bot Error: TON price API request failed - Error: {$error}");
        if ($cachedRate !== null) {
            return $cachedRate;
        }
        throw new RuntimeException('TON price request failed.');
    }

    $data = json_decode($response, true);

    if (!is_array($data) || !isset($data['the-open-network']['usd'])) {
        error_log("Telegram Bot Error: Invalid TON price response - " . substr($response ?: '', 0, 200));
        if ($cachedRate !== null) {
            return $cachedRate;
        }
        throw new RuntimeException('Invalid TON price response.');
    }

    $rate = (float)$data['the-open-network']['usd'];

    try {
        $payload = json_encode(['rate' => $rate, 'timestamp' => time()], JSON_THROW_ON_ERROR);
        @file_put_contents($cacheFile, $payload, LOCK_EX);
    } catch (\JsonException $e) {
        // تجاهل أخطاء التخزين المؤقت بصمت
    }

    return $rate;
}

function formatDecimal(float $value): string
{
    $formatted = number_format($value, 6, '.', '');
    return rtrim(rtrim($formatted, '0'), '.') ?: '0';
}

function formatStars(float $value): string
{
    return (string)max(0, (int)round($value));
}

function sendMessage(string $endpoint, int $chatId, string $text, array $options = []): void
{
    $payload = array_merge([
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML',
    ], $options);

    $ch = curl_init($endpoint);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $error || ($httpCode !== 200 && $httpCode !== 0)) {
        error_log("Telegram Bot Error: Failed to send message - HTTP: {$httpCode}, Error: {$error}, Response: " . substr($response ?: '', 0, 200));
    }
}

function answerCallbackQuery(string $endpoint, string $callbackId, string $text = ''): void
{
    if ($callbackId === '') {
        return;
    }

    $url = str_replace('/sendMessage', '/answerCallbackQuery', $endpoint);
    $payload = [
        'callback_query_id' => $callbackId,
    ];
    
    if ($text !== '') {
        $payload['text'] = $text;
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $error || ($httpCode !== 200 && $httpCode !== 0)) {
        error_log("Telegram Bot Error: Failed to answer callback query - HTTP: {$httpCode}, Error: {$error}, Response: " . substr($response ?: '', 0, 200));
    }
}

function buildAdminPanelKeyboard(): array
{
    return [
        'inline_keyboard' => [
            [
                ['text' => '📊 الإحصائيات', 'callback_data' => 'admin_stats'],
                ['text' => '👑 إدارة الأدمن', 'callback_data' => 'admin_manage_admins'],
            ],
            [
                ['text' => '👥 عرض المستخدمين', 'callback_data' => 'admin_view_users'],
                ['text' => '💬 عرض الكروبات', 'callback_data' => 'admin_view_groups'],
            ],
            [
                ['text' => '📢 القنوات العامة', 'callback_data' => 'admin_private_channels'],
                ['text' => '👥 قنوات الكروبات', 'callback_data' => 'admin_group_channels'],
            ],
            [
                ['text' => '🔔 الإشعارات', 'callback_data' => 'admin_notifications'],
                ['text' => '🔄 تحديث', 'callback_data' => 'admin_refresh'],
            ],
            [
                ['text' => '❌ إغلاق', 'callback_data' => 'admin_close'],
            ],
        ],
    ];
}

function handleAdminInlineButtons(string $endpoint, int $chatId, string $data, ?int $senderId, ?int $messageId, string $botToken, array $update): bool
{
    $callbackQuery = $update['callback_query'] ?? null;
    if (!$callbackQuery) {
        return false;
    }
    
    answerCallbackQuery($endpoint, $callbackQuery['id'] ?? '');
    
    switch ($data) {
        case 'admin_stats':
            $adminData = loadAdminData();
            $users = loadUsersData();
            $groups = loadGroupsData();
            $groupChannels = loadGroupChannelsData();
            
            $stats = "📊 <b>إحصائيات البوت</b>\n\n";
            $stats .= "👥 <b>عدد المستخدمين:</b> " . count($users) . "\n";
            $stats .= "👥 <b>عدد الكروبات:</b> " . count($groups) . "\n";
            $stats .= "📢 <b>عدد القنوات العامة:</b> " . count($adminData['private_channels'] ?? []) . "\n";
            $stats .= "📢 <b>عدد قنوات الكروبات:</b> " . count($groupChannels) . "\n";
            $stats .= "👑 <b>عدد الأدمن:</b> " . (count($adminData['admins'] ?? []) + (isset($adminData['owner']) ? 1 : 0)) . "\n";
            $stats .= "🔔 <b>الإشعارات:</b> " . (($adminData['notifications_enabled'] ?? false) ? '✅ مفعلة' : '❌ معطلة');
            
            editMessageText($endpoint, $chatId, $messageId, $stats, buildAdminPanelKeyboard());
            return true;
            
        case 'admin_manage_admins':
            $adminData = loadAdminData();
            $admins = $adminData['admins'] ?? [];
            $owner = $adminData['owner'] ?? null;
            
            $text = "👑 <b>إدارة الأدمن</b>\n\n";
            if ($owner !== null) {
                $text .= "👑 <b>المالك:</b> " . $owner . "\n";
            }
            $text .= "👥 <b>عدد الأدمن:</b> " . count($admins) . "\n\n";
            $text .= "<b>الأدمن:</b>\n";
            if (empty($admins)) {
                $text .= "لا يوجد أدمن حالياً\n\n";
            } else {
                foreach ($admins as $adminId) {
                    $text .= "• " . $adminId . "\n";
                }
            }
            $text .= "\n<b>الأوامر:</b>\n";
            $text .= "/addadmin [user_id] - إضافة أدمن\n";
            $text .= "/deladmin [user_id] - حذف أدمن";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_private_channels':
            $adminData = loadAdminData();
            $channels = $adminData['private_channels'] ?? [];
            
            $text = "📢 <b>القنوات العامة</b>\n\n";
            $text .= "📊 <b>عدد القنوات:</b> " . count($channels) . "\n\n";
            if (empty($channels)) {
                $text .= "لا توجد قنوات حالياً\n\n";
            } else {
                $text .= "<b>القنوات:</b>\n";
                foreach ($channels as $channel) {
                    $text .= "• @" . ltrim($channel, '@') . "\n";
                }
            }
            $text .= "\n<b>الأوامر:</b>\n";
            $text .= "/addchannel [channel_id] - إضافة قناة\n";
            $text .= "/delchannel [channel_id] - حذف قناة";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_group_channels':
            $groupChannels = loadGroupChannelsData();
            
            $text = "👥 <b>قنوات الكروبات</b>\n\n";
            $text .= "📊 <b>عدد القنوات:</b> " . count($groupChannels) . "\n\n";
            if (empty($groupChannels)) {
                $text .= "لا توجد قنوات حالياً\n\n";
            } else {
                $text .= "<b>القنوات:</b>\n";
                foreach ($groupChannels as $channel) {
                    $text .= "• @" . ltrim($channel, '@') . "\n";
                }
            }
            $text .= "\n<b>الأوامر:</b>\n";
            $text .= "/addgroupchannel [channel_id] - إضافة قناة\n";
            $text .= "/delgroupchannel [channel_id] - حذف قناة";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_view_users':
            $users = loadUsersData();
            
            $text = "👥 <b>قائمة المستخدمين</b>\n\n";
            $text .= "📊 <b>إجمالي المستخدمين:</b> " . count($users) . "\n\n";
            
            if (empty($users)) {
                $text .= "لا يوجد مستخدمين حالياً";
            } else {
                $text .= "<b>المستخدمين:</b>\n\n";
                $counter = 1;
                foreach ($users as $userId => $userData) {
                    $name = ($userData['first_name'] ?? '') . ' ' . ($userData['last_name'] ?? '');
                    $name = trim($name) ?: 'غير معروف';
                    $username = isset($userData['username']) ? '@' . $userData['username'] : 'لا يوجد';
                    $joinedDate = isset($userData['joined_at']) ? date('Y-m-d H:i', $userData['joined_at']) : 'غير معروف';
                    
                    $text .= "<b>{$counter}.</b> {$name}\n";
                    $text .= "🆔 ID: <code>{$userId}</code>\n";
                    $text .= "📱 Username: {$username}\n";
                    $text .= "📅 تاريخ الانضمام: {$joinedDate}\n\n";
                    
                    $counter++;
                    
                    // تقسيم الرسالة إذا كانت طويلة (حد 4000 حرف)
                    if (mb_strlen($text) > 3800) {
                        editMessageText($endpoint, $chatId, $messageId, $text, [
                            'inline_keyboard' => [
                                [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                            ],
                        ]);
                        $text = "👥 <b>قائمة المستخدمين (متابعة)</b>\n\n";
                    }
                }
            }
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_view_groups':
            $groups = loadGroupsData();
            
            $text = "💬 <b>قائمة الكروبات</b>\n\n";
            $text .= "📊 <b>إجمالي الكروبات:</b> " . count($groups) . "\n\n";
            
            if (empty($groups)) {
                $text .= "لا توجد كروبات حالياً";
            } else {
                $text .= "<b>الكروبات:</b>\n\n";
                $counter = 1;
                foreach ($groups as $groupId => $groupData) {
                    $title = $groupData['title'] ?? 'غير معروف';
                    $type = $groupData['type'] ?? 'group';
                    $addedDate = isset($groupData['added_at']) ? date('Y-m-d H:i', $groupData['added_at']) : 'غير معروف';
                    
                    $text .= "<b>{$counter}.</b> {$title}\n";
                    $text .= "🆔 ID: <code>{$groupId}</code>\n";
                    $text .= "📌 النوع: {$type}\n";
                    $text .= "📅 تاريخ الإضافة: {$addedDate}\n\n";
                    
                    $counter++;
                    
                    // تقسيم الرسالة إذا كانت طويلة (حد 4000 حرف)
                    if (mb_strlen($text) > 3800) {
                        editMessageText($endpoint, $chatId, $messageId, $text, [
                            'inline_keyboard' => [
                                [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                            ],
                        ]);
                        $text = "💬 <b>قائمة الكروبات (متابعة)</b>\n\n";
                    }
                }
            }
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_notifications':
            $adminData = loadAdminData();
            $enabled = $adminData['notifications_enabled'] ?? false;
            
            $text = "🔔 <b>إعدادات الإشعارات</b>\n\n";
            $text .= "الحالة الحالية: " . ($enabled ? '✅ مفعلة' : '❌ معطلة') . "\n\n";
            $text .= "<b>الأوامر:</b>\n";
            $text .= "/notifications on - تفعيل\n";
            $text .= "/notifications off - تعطيل";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => ($enabled ? '❌ تعطيل' : '✅ تفعيل'), 'callback_data' => 'admin_toggle_notifications'],
                    ],
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_toggle_notifications':
            $adminData = loadAdminData();
            $adminData['notifications_enabled'] = !($adminData['notifications_enabled'] ?? false);
            saveAdminData($adminData);
            
            $text = "✅ تم " . ($adminData['notifications_enabled'] ? 'تفعيل' : 'تعطيل') . " الإشعارات بنجاح";
            answerCallbackQuery($endpoint, $callbackQuery['id'] ?? '', $text);
            
            // تحديث الرسالة
            $enabled = $adminData['notifications_enabled'];
            $text = "🔔 <b>إعدادات الإشعارات</b>\n\n";
            $text .= "الحالة الحالية: " . ($enabled ? '✅ مفعلة' : '❌ معطلة') . "\n\n";
            $text .= "<b>الأوامر:</b>\n";
            $text .= "/notifications on - تفعيل\n";
            $text .= "/notifications off - تعطيل";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => ($enabled ? '❌ تعطيل' : '✅ تفعيل'), 'callback_data' => 'admin_toggle_notifications'],
                    ],
                    [['text' => '🔙 رجوع', 'callback_data' => 'admin_back']],
                ],
            ];
            editMessageText($endpoint, $chatId, $messageId, $text, $keyboard);
            return true;
            
        case 'admin_refresh':
            $adminData = loadAdminData();
            $users = loadUsersData();
            $groups = loadGroupsData();
            $groupChannels = loadGroupChannelsData();
            
            $stats = "📊 <b>لوحة التحكم الرئيسية</b>\n\n";
            $stats .= "👥 <b>عدد المستخدمين:</b> " . count($users) . "\n";
            $stats .= "👥 <b>عدد الكروبات:</b> " . count($groups) . "\n";
            $stats .= "📢 <b>عدد القنوات العامة:</b> " . count($adminData['private_channels'] ?? []) . "\n";
            $stats .= "📢 <b>عدد قنوات الكروبات:</b> " . count($groupChannels) . "\n";
            $stats .= "🔔 <b>الإشعارات:</b> " . (($adminData['notifications_enabled'] ?? false) ? '✅ مفعلة' : '❌ معطلة') . "\n";
            $stats .= "👑 <b>عدد الأدمن:</b> " . (count($adminData['admins'] ?? []) + (isset($adminData['owner']) ? 1 : 0)) . "\n\n";
            $stats .= "اختر من القائمة أدناه:";
            
            answerCallbackQuery($endpoint, $callbackQuery['id'] ?? '', "✅ تم التحديث");
            editMessageText($endpoint, $chatId, $messageId, $stats, buildAdminPanelKeyboard());
            return true;
            
        case 'admin_back':
            $adminData = loadAdminData();
            $users = loadUsersData();
            $groups = loadGroupsData();
            $groupChannels = loadGroupChannelsData();
            
            $stats = "📊 <b>لوحة التحكم الرئيسية</b>\n\n";
            $stats .= "👥 <b>عدد المستخدمين:</b> " . count($users) . "\n";
            $stats .= "👥 <b>عدد الكروبات:</b> " . count($groups) . "\n";
            $stats .= "📢 <b>عدد القنوات العامة:</b> " . count($adminData['private_channels'] ?? []) . "\n";
            $stats .= "📢 <b>عدد قنوات الكروبات:</b> " . count($groupChannels) . "\n";
            $stats .= "🔔 <b>الإشعارات:</b> " . (($adminData['notifications_enabled'] ?? false) ? '✅ مفعلة' : '❌ معطلة') . "\n";
            $stats .= "👑 <b>عدد الأدمن:</b> " . (count($adminData['admins'] ?? []) + (isset($adminData['owner']) ? 1 : 0)) . "\n\n";
            $stats .= "اختر من القائمة أدناه:";
            
            editMessageText($endpoint, $chatId, $messageId, $stats, buildAdminPanelKeyboard());
            return true;
            
        case 'admin_close':
            deleteMessage($endpoint, $chatId, $messageId);
            return true;
    }
    
    return false;
}

function editMessageText(string $endpoint, int $chatId, ?int $messageId, string $text, array $keyboard = []): void
{
    if ($messageId === null) {
        return;
    }
    
    $url = str_replace('/sendMessage', '/editMessageText', $endpoint);
    $payload = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $text,
        'parse_mode' => 'HTML',
    ];
    
    if (!empty($keyboard)) {
        $payload['reply_markup'] = json_encode($keyboard, JSON_UNESCAPED_UNICODE);
    }
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false || $error || ($httpCode !== 200 && $httpCode !== 0)) {
        error_log("Telegram Bot Error: Failed to edit message - HTTP: {$httpCode}, Error: {$error}, Response: " . substr($response ?: '', 0, 200));
    }
}

function deleteMessage(string $endpoint, int $chatId, ?int $messageId): void
{
    if ($messageId === null) {
        return;
    }
    
    $url = str_replace('/sendMessage', '/deleteMessage', $endpoint);
    $payload = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
    ];
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_exec($ch);
    curl_close($ch);
}

// ==================== Admin System Functions ====================

function initAdminData(): void
{
    static $initialized = false;
    if ($initialized) {
        return;
    }
    
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    
    if (!is_file(ADMIN_DATA_FILE)) {
        // لإضافة معرف أدمن مباشرة، غيّر null إلى معرفك (مثال: 123456789)
        // أو أضف معرفات في المصفوفة admins (مثال: 'admins' => [123456789, 987654321])
        if (!is_dir(ADMIN_DATA_DIR)) {
            @mkdir(ADMIN_DATA_DIR, 0777, true);
        }
        @file_put_contents(ADMIN_DATA_FILE, json_encode(['owner' => 6565689498, 'admins' => [], 'notifications_enabled' => false, 'private_channels' => [], 'group_channels' => []], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    if (!is_file(USERS_DATA_FILE)) {
        if (!is_dir(ADMIN_DATA_DIR)) {
            @mkdir(ADMIN_DATA_DIR, 0777, true);
        }
        @file_put_contents(USERS_DATA_FILE, json_encode([], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    if (!is_file(GROUPS_DATA_FILE)) {
        if (!is_dir(ADMIN_DATA_DIR)) {
            @mkdir(ADMIN_DATA_DIR, 0777, true);
        }
        @file_put_contents(GROUPS_DATA_FILE, json_encode([], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    if (!is_file(CHANNELS_DATA_FILE)) {
        if (!is_dir(ADMIN_DATA_DIR)) {
            @mkdir(ADMIN_DATA_DIR, 0777, true);
        }
        @file_put_contents(CHANNELS_DATA_FILE, json_encode([], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    if (!is_file(GROUP_CHANNELS_FILE)) {
        if (!is_dir(ADMIN_DATA_DIR)) {
            @mkdir(ADMIN_DATA_DIR, 0777, true);
        }
        @file_put_contents(GROUP_CHANNELS_FILE, json_encode([], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    $initialized = true;
}

function loadAdminData(): array
{
    if (!is_readable(ADMIN_DATA_FILE)) {
        return ['owner' => null, 'admins' => [], 'notifications_enabled' => false, 'private_channels' => [], 'group_channels' => []];
    }
    $data = @file_get_contents(ADMIN_DATA_FILE);
    if ($data === false) {
        return ['owner' => null, 'admins' => [], 'notifications_enabled' => false, 'private_channels' => [], 'group_channels' => []];
    }
    $decoded = json_decode($data, true);
    return is_array($decoded) ? $decoded : ['owner' => null, 'admins' => [], 'notifications_enabled' => false, 'private_channels' => [], 'group_channels' => []];
}

function saveAdminData(array $data): void
{
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    @file_put_contents(ADMIN_DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

function loadUsersData(): array
{
    if (!is_readable(USERS_DATA_FILE)) {
        return [];
    }
    $data = @file_get_contents(USERS_DATA_FILE);
    if ($data === false) {
        return [];
    }
    $decoded = json_decode($data, true);
    return is_array($decoded) ? $decoded : [];
}

function saveUsersData(array $data): void
{
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    @file_put_contents(USERS_DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

function loadGroupsData(): array
{
    if (!is_readable(GROUPS_DATA_FILE)) {
        return [];
    }
    $data = @file_get_contents(GROUPS_DATA_FILE);
    if ($data === false) {
        return [];
    }
    $decoded = json_decode($data, true);
    return is_array($decoded) ? $decoded : [];
}

function saveGroupsData(array $data): void
{
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    @file_put_contents(GROUPS_DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

function loadChannelsData(): array
{
    if (!is_readable(CHANNELS_DATA_FILE)) {
        return [];
    }
    $data = @file_get_contents(CHANNELS_DATA_FILE);
    if ($data === false) {
        return [];
    }
    $decoded = json_decode($data, true);
    return is_array($decoded) ? $decoded : [];
}

function saveChannelsData(array $data): void
{
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    @file_put_contents(CHANNELS_DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

function loadGroupChannelsData(): array
{
    if (!is_readable(GROUP_CHANNELS_FILE)) {
        return [];
    }
    $data = @file_get_contents(GROUP_CHANNELS_FILE);
    if ($data === false) {
        return [];
    }
    $decoded = json_decode($data, true);
    return is_array($decoded) ? $decoded : [];
}

function saveGroupChannelsData(array $data): void
{
    if (!is_dir(ADMIN_DATA_DIR)) {
        @mkdir(ADMIN_DATA_DIR, 0777, true);
    }
    @file_put_contents(GROUP_CHANNELS_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

function isOwner(int $userId): bool
{
    $adminData = loadAdminData();
    return isset($adminData['owner']) && $adminData['owner'] === $userId;
}

function isAdmin(int $userId): bool
{
    $adminData = loadAdminData();
    return isOwner($userId) || (isset($adminData['admins']) && in_array($userId, $adminData['admins'], true));
}

function addUser(int $userId, array $userInfo = []): void
{
    $users = loadUsersData();
    if (!isset($users[$userId])) {
        $users[$userId] = array_merge([
            'id' => $userId,
            'username' => $userInfo['username'] ?? null,
            'first_name' => $userInfo['first_name'] ?? null,
            'last_name' => $userInfo['last_name'] ?? null,
            'joined_at' => time(),
        ], $userInfo);
        saveUsersData($users);
    }
}

function addGroup(int $groupId, array $groupInfo = []): void
{
    $groups = loadGroupsData();
    if (!isset($groups[$groupId])) {
        $groups[$groupId] = array_merge([
            'id' => $groupId,
            'title' => $groupInfo['title'] ?? null,
            'type' => $groupInfo['type'] ?? 'group',
            'added_at' => time(),
        ], $groupInfo);
        saveGroupsData($groups);
    }
}

function checkChannelSubscription(string $botToken, int $userId, array $channels): array
{
    $notSubscribed = [];
    if (empty($channels) || $userId === null) {
        return $notSubscribed;
    }
    
    foreach ($channels as $channelId) {
        try {
            $url = "https://api.telegram.org/bot{$botToken}/getChatMember";
            $ch = curl_init($url);
            if ($ch === false) {
                continue;
            }
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, [
                'chat_id' => $channelId,
                'user_id' => $userId,
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 3);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            $response = curl_exec($ch);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($response && empty($error)) {
                $data = json_decode($response, true);
                if (is_array($data) && isset($data['result']['status'])) {
                    $status = $data['result']['status'];
                    if (!in_array($status, ['member', 'administrator', 'creator'], true)) {
                        $notSubscribed[] = $channelId;
                    }
                } else {
                    // في حالة الخطأ، نعتبر أن المستخدم غير مشترك لتجنب المشاكل
                    $notSubscribed[] = $channelId;
                }
            } else {
                // في حالة فشل الاتصال، نتجاهل هذا القناة ونكمل
                continue;
            }
        } catch (Exception $e) {
            // تجاهل الأخطاء ونكمل
            continue;
        }
    }
    return $notSubscribed;
}

function sendNotificationToOwner(string $botToken, string $message): void
{
    $adminData = loadAdminData();
    if (isset($adminData['owner']) && ($adminData['notifications_enabled'] ?? false) === true) {
        $endpoint = "https://api.telegram.org/bot{$botToken}/sendMessage";
        sendMessage($endpoint, $adminData['owner'], $message);
    }
}

function handleAdminCommands(string $endpoint, int $chatId, string $text, ?int $senderId, ?int $messageId, string $botToken, array $update): bool
{
    if ($senderId === null) {
        return false;
    }
    
    $text = trim($text);
    $parts = explode(' ', $text, 2);
    $command = mb_strtolower($parts[0]);
    
    // تعيين المالك (متاح لأي شخص إذا لم يكن هناك مالك)
    if ($command === '/setowner') {
        $adminData = loadAdminData();
        if (!isset($adminData['owner']) || $adminData['owner'] === null) {
            $adminData['owner'] = $senderId;
            saveAdminData($adminData);
            sendMessage($endpoint, $chatId, "✅ تم تعيينك كمالك للبوت.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        } else {
            sendMessage($endpoint, $chatId, "⚠️ المالك موجود بالفعل.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // التحقق من صلاحيات الأدمن للأوامر الأخرى
    if (!isAdmin($senderId)) {
        return false;
    }
    
    // لوحة التحكم
    if ($command === '/admin' || $command === '/panel') {
        $adminData = loadAdminData();
        $users = loadUsersData();
        $groups = loadGroupsData();
        $groupChannels = loadGroupChannelsData();
        
        $stats = "📊 <b>لوحة التحكم الرئيسية</b>\n\n";
        $stats .= "👥 <b>عدد المستخدمين:</b> " . count($users) . "\n";
        $stats .= "👥 <b>عدد الكروبات:</b> " . count($groups) . "\n";
        $stats .= "📢 <b>عدد القنوات العامة:</b> " . count($adminData['private_channels'] ?? []) . "\n";
        $stats .= "📢 <b>عدد قنوات الكروبات:</b> " . count($groupChannels) . "\n";
        $stats .= "🔔 <b>الإشعارات:</b> " . (($adminData['notifications_enabled'] ?? false) ? '✅ مفعلة' : '❌ معطلة') . "\n";
        $stats .= "👑 <b>عدد الأدمن:</b> " . (count($adminData['admins'] ?? []) + (isset($adminData['owner']) ? 1 : 0)) . "\n\n";
        $stats .= "اختر من القائمة أدناه:";
        
        $options = ['reply_markup' => json_encode(buildAdminPanelKeyboard(), JSON_UNESCAPED_UNICODE)];
        if ($messageId !== null) {
            $options['reply_to_message_id'] = $messageId;
        }
        sendMessage($endpoint, $chatId, $stats, $options);
        return true;
    }
    
    // إضافة أدمن
    if ($command === '/addadmin' && isset($parts[1])) {
        $newAdminId = (int)trim($parts[1]);
        if ($newAdminId > 0) {
            $adminData = loadAdminData();
            if (!isset($adminData['admins'])) {
                $adminData['admins'] = [];
            }
            if (!in_array($newAdminId, $adminData['admins'], true)) {
                $adminData['admins'][] = $newAdminId;
                saveAdminData($adminData);
                sendMessage($endpoint, $chatId, "✅ تم إضافة الأدمن بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذا المستخدم أدمن بالفعل.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف المستخدم بشكل صحيح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // حذف أدمن
    if ($command === '/deladmin' && isset($parts[1])) {
        $adminId = (int)trim($parts[1]);
        if ($adminId > 0) {
            $adminData = loadAdminData();
            if (isset($adminData['admins']) && ($key = array_search($adminId, $adminData['admins'], true)) !== false) {
                unset($adminData['admins'][$key]);
                $adminData['admins'] = array_values($adminData['admins']);
                saveAdminData($adminData);
                sendMessage($endpoint, $chatId, "✅ تم حذف الأدمن بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذا المستخدم ليس أدمن.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف المستخدم بشكل صحيح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // إضافة قناة عامة
    if ($command === '/addchannel' && isset($parts[1])) {
        $channelId = trim($parts[1]);
        if ($channelId !== '') {
            $adminData = loadAdminData();
            if (!isset($adminData['private_channels'])) {
                $adminData['private_channels'] = [];
            }
            if (!in_array($channelId, $adminData['private_channels'], true)) {
                $adminData['private_channels'][] = $channelId;
                saveAdminData($adminData);
                sendMessage($endpoint, $chatId, "✅ تم إضافة القناة العامة بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذه القناة موجودة بالفعل.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف القناة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // حذف قناة عامة
    if ($command === '/delchannel' && isset($parts[1])) {
        $channelId = trim($parts[1]);
        if ($channelId !== '') {
            $adminData = loadAdminData();
            if (isset($adminData['private_channels']) && ($key = array_search($channelId, $adminData['private_channels'], true)) !== false) {
                unset($adminData['private_channels'][$key]);
                $adminData['private_channels'] = array_values($adminData['private_channels']);
                saveAdminData($adminData);
                sendMessage($endpoint, $chatId, "✅ تم حذف القناة العامة بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذه القناة غير موجودة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف القناة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // إضافة قناة للكروبات
    if ($command === '/addgroupchannel' && isset($parts[1])) {
        $channelId = trim($parts[1]);
        if ($channelId !== '') {
            $groupChannels = loadGroupChannelsData();
            if (!in_array($channelId, $groupChannels, true)) {
                $groupChannels[] = $channelId;
                saveGroupChannelsData($groupChannels);
                sendMessage($endpoint, $chatId, "✅ تم إضافة قناة الكروبات بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذه القناة موجودة بالفعل.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف القناة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // حذف قناة للكروبات
    if ($command === '/delgroupchannel' && isset($parts[1])) {
        $channelId = trim($parts[1]);
        if ($channelId !== '') {
            $groupChannels = loadGroupChannelsData();
            if (($key = array_search($channelId, $groupChannels, true)) !== false) {
                unset($groupChannels[$key]);
                $groupChannels = array_values($groupChannels);
                saveGroupChannelsData($groupChannels);
                sendMessage($endpoint, $chatId, "✅ تم حذف قناة الكروبات بنجاح.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            } else {
                sendMessage($endpoint, $chatId, "⚠️ هذه القناة غير موجودة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
            }
        } else {
            sendMessage($endpoint, $chatId, "⚠️ الرجاء إدخال معرف القناة.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // تفعيل/تعطيل الإشعارات
    if ($command === '/notifications' && isset($parts[1])) {
        $action = mb_strtolower(trim($parts[1]));
        $adminData = loadAdminData();
        if ($action === 'on' || $action === 'تفعيل') {
            $adminData['notifications_enabled'] = true;
            saveAdminData($adminData);
            sendMessage($endpoint, $chatId, "✅ تم تفعيل الإشعارات.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        } elseif ($action === 'off' || $action === 'تعطيل') {
            $adminData['notifications_enabled'] = false;
            saveAdminData($adminData);
            sendMessage($endpoint, $chatId, "❌ تم تعطيل الإشعارات.", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        } else {
            sendMessage($endpoint, $chatId, "⚠️ استخدم: /notifications on أو /notifications off", $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        }
        return true;
    }
    
    // الإحصائيات
    if ($command === '/stats') {
        $users = loadUsersData();
        $groups = loadGroupsData();
        $adminData = loadAdminData();
        $groupChannels = loadGroupChannelsData();
        
        $stats = "📊 <b>إحصائيات البوت</b>\n\n";
        $stats .= "👥 <b>عدد المستخدمين:</b> " . count($users) . "\n";
        $stats .= "👥 <b>عدد الكروبات:</b> " . count($groups) . "\n";
        $stats .= "📢 <b>عدد القنوات العامة:</b> " . count($adminData['private_channels'] ?? []) . "\n";
        $stats .= "📢 <b>عدد قنوات الكروبات:</b> " . count($groupChannels) . "\n";
        $stats .= "👑 <b>عدد الأدمن:</b> " . (count($adminData['admins'] ?? []) + (isset($adminData['owner']) ? 1 : 0)) . "\n";
        
        sendMessage($endpoint, $chatId, $stats, $messageId !== null ? ['reply_to_message_id' => $messageId] : []);
        return true;
    }
    
    return false;
}

